/*
 * oc_ema_mgmt_at.c
 *
 * Created: 6/24/2019 11:49:48 AM
 *  Author: matt.voigt
 */ 

 #include "oc_ema_mgmt_at.h"

/*########## ema Mgmt At cmds ##########*/
static const char ema_atOKPattern[]     = "\r\nOK\r\n";

static const char ema_atCmdAT[]         = "at\r"; 
static const char ema_atDisEchoCmd[]    = "ate0\r";
static const char ema_atEnEchoCmd[]     = "ate1\r";
static const char ema_atGetFWVer[]      = "at+gmr\r";
static const char ema_atGetSN[]         = "at+gsn\r";
static const char ema_atSetBID[]        = "at+ocbid=";
static const char ema_atGetBID[]        = "at+ocbid?\r";
static const char ema_atTestBID[]       = "at+ocbid=?\r";
static const char ema_atListCMDs[]      = "at+clac\r";

/*###########################################*/

/*########### ema Mgmt URCs & codes ###########*/
static const char ema_URCPattern[] = "OCURC:";

/*###########################################*/


/*############ Helper Functions ############*/
/* 
CheckForMsg: 
Helper function to check an input buffer(*buff) for a valid msg. A valid msg begins and ends with 
"\r\n". The address of the the valid msg is returned(int). This function can be used on partial msgs, 
as long as the *loc variable is updated prior to calling it. 
*/
int CheckForMsg( char *buff, unsigned short len, int *loc )
{ 
    int i;
    char *msg = 0;
    unsigned char r = 0;
    unsigned char n = 0;
    unsigned char msgfound = 0;

    // check for a valid msg containg two pairs of "\r\n"
    for ( i=*loc; i<len; i++ ) {
        if ( *(buff+i) == '\r' ) {
            r++;
        }else if ( *(buff+i) == '\n' ) {
            n++;
        }
        if ( r == 2 && n == 2 ) {
            msgfound = 1;
            break;
        }
    }

    if ( msgfound ) {
        // get a pointer to the beginning of the msg
        msg = strstr( buff+*loc, "\r\n" );
        *loc = i;
    }
    return (int)msg;
}

/* 
ParseMsg: 
Helper function to parse a valid msg(msg_ptr) and check that it's a URC. The URC ascii format 
is also converted to hex and returned(char). 
*/
static char ParseMsg( char *msg_ptr )
{
    char val = MGMT_URC_INVALID;
    char *p;
    char ch[2];

    // check the string for a non urc msg
    /*if ( strstr( (char*)msg_ptr, ema_ATReady ) ) {
        // found non urc match
        val = MGMT_URC_AT_READY;
    }else*/ if ( strstr( (char*)msg_ptr, ema_URCPattern ) ) {
        // found urc, convert the ascii values to integer
        p = strstr( (char*)msg_ptr, ": " );
        ch[0] = *(p+2);
        ch[1] = *(p+3);
        val = atoi( ch );
    }

    return  val;
}

/* 
emaModemAt_build_cmd: 
Copies a cmd(*cmd) to the ema at tx buffer(context->at_cmd). 
*/
static void emaMgmtAT_BuildCmd( struct ema_mgmt_context *context, const char *cmd )
{
    sprintf(context->at_cmd, "%s", cmd);
}

/* 
emaMgmtURC_GetValues: 
Checks the input buffer(*buff) up to an input length(len) for a valid URC. 
The URC integer value is returned. 
*/
int emaMgmtURC_GetValue( char *msg, unsigned short len )
{
    int val = 0;

    // null terminate the entire buffer for string ops
    *(msg+len) = 0;

    val = ParseMsg( msg );

    return val;
}
/*###########################################*/

/*###########################################*/

/*############ ema Mgmt AT cmd APIs ############*/
/* 
emaModemAT_init: 
This function is provided for the users application to pass in a function pointer(*func_ptr) for a specific AT cmd 
send and recieve function handler. Subsequent APIs in this file will use this function pointer for AT transactions. 
*/
void emaMgmtAT_Init( int (*func_ptr)(struct ema_mgmt_context *), struct ema_mgmt_context *handle )
{
    // set the user provided function for an at query
    ema_at_comm_handle = func_ptr;

    // reset vars
    handle->at_cmd_flags = 0;
    handle->response_flags = 0;
    handle->urc_status = 0;
}

/* 
emaModemAT_check_comm: 
Sends a simple AT cmd to ema for communication validation. The response is checked accordingly 
*/
int emaMgmtAT_CheckComm( struct ema_mgmt_context *context )
{
    int st;
    unsigned short len;

    emaMgmtAT_BuildCmd( context, ema_atCmdAT );

    // call the user provided hook for at comms
    st = (*ema_at_comm_handle)( context );
    if ( st > 0 ) {
        // check the response
        st = emaMgmtAT_CheckOK( context->response, &len );
    }

    return st;
}

/* 
emaMgmtAT_GetFWVersion: 
 
*/
int emaMgmtAT_GetFWVersion( struct ema_mgmt_context *context, char *fw_string )
{
    int st, msg, loc;
    unsigned short len;

    emaMgmtAT_BuildCmd(context, ema_atGetFWVer);

    // call the user provided hook for at comms
    st = (*ema_at_comm_handle)( context );
    if ( st > 0 ) {
        // check the response
        st = emaMgmtAT_CheckOK( context->response, &len );
        if ( st == MGMT_AT_ST_SUCCESS ) {
            // get the fw string
            loc = 0;
            msg = CheckForMsg( context->response, len, &loc );
            if ( msg ) {
                // null terminate for string ops, and update the output buff, skip "\r\n"
                *((char*)msg+len-2) = 0;
                strcpy( fw_string, (char*)msg+2 );
            }
        }
    }
    return st;
}

/* 
emaMgmtAT_ListAvailCMDs: 
 
*/
int emaMgmtAT_ListAvailCMDs( struct ema_mgmt_context *context, char *cmd_list )
{
    int st, msg, loc;
    unsigned short len;

    emaMgmtAT_BuildCmd(context, ema_atListCMDs);

    // call the user provided hook for at comms
    st = (*ema_at_comm_handle)( context );
    if ( st > 0 ) {
        // check the response
        st = emaMgmtAT_CheckOK( context->response, &len );
        if ( st == MGMT_AT_ST_SUCCESS ) {
            // get the clac string
            loc = 0;
            msg = CheckForMsg( context->response, len, &loc );
            if ( msg ) {
                // null terminate for string ops, and update the output buff, skip "\r\n"
                *((char*)msg+len) = 0;
                strcpy( cmd_list, (char*)msg );
            }
        }
    }
    return st;
}

/* 
emaMgmtAT_GetSN: 
 
*/
int emaMgmtAT_GetSN( struct ema_mgmt_context *context, char *sn_string )
{
    int st, msg, loc;
    unsigned short len;

    emaMgmtAT_BuildCmd(context, ema_atGetSN);

    // call the user provided hook for at comms
    st = (*ema_at_comm_handle)( context );
    if ( st > 0 ) {
        // check the response
        st = emaMgmtAT_CheckOK( context->response, &len );
        if ( st == MGMT_AT_ST_SUCCESS ) {
            // get the sn string
            loc = 0;
            msg = CheckForMsg( context->response, len, &loc );
            if ( msg ) {
                // null terminate for string ops, and update the output buff, skip "\r\n"
                *((char*)msg+len-2) = 0;
                strcpy( sn_string, (char*)msg+2 );
            }
        }
    }
    return st;
}

/* 
emaMgmtAT_SetBID: 
Sets the Board ID in ema. The board id string(*bid_string) must be a null terminated string. 
*/
int emaMgmtAT_SetBID( struct ema_mgmt_context *context, char *bid_string )
{
    int st, l;
    unsigned short len;

    l = sprintf( context->at_cmd, "%s", ema_atSetBID );
    l += sprintf( context->at_cmd+l, "\"%s\"", bid_string );
    sprintf( context->at_cmd+l, "\r" );

    // call the user provided hook for at comms
    st = (*ema_at_comm_handle)( context );
    if ( st > 0 ) {
        // check the response
        st = emaMgmtAT_CheckOK( context->response, &len );
    }
    return st;
}

/* 
emaMgmtAT_GetBID: 
 
*/
int emaMgmtAT_GetBID( struct ema_mgmt_context *context, char *bid_string )
{
    int st, msg, loc;
    unsigned short len;

    emaMgmtAT_BuildCmd(context, ema_atGetBID);

    // call the user provided hook for at comms
    st = (*ema_at_comm_handle)( context );
    if ( st > 0 ) {
        // check the response
        st = emaMgmtAT_CheckOK( context->response, &len );
        if ( st == MGMT_AT_ST_SUCCESS ) {
            // get the bid string
            loc = 0;
            msg = CheckForMsg( context->response, len, &loc );
            if ( msg ) {
                // validate "+OCBID:" string in msg
                if ( strstr( context->response, "+OCBID:" ) ) {
                    // null terminate for string ops, and update the output buff, skip "\r\n+OCBID: ", and quotes around BID string
                    *((char*)msg+len-3) = 0;
                    strcpy( bid_string, (char*)msg+11 );
                }else{
                    st = MGMT_AT_ST_BAD_RESPONSE;
                }
            }
        }
    }
    return st;
}

/* 
emaMgmtAT_BIDTestCmd: 
*/
int emaMgmtAT_BIDTestCmd( struct ema_mgmt_context *context, char *formatting )
{
    int st, msg, loc;
    unsigned short len;

    emaMgmtAT_BuildCmd(context, ema_atTestBID);

    // call the user provided hook for at comms
    st = (*ema_at_comm_handle)( context );
    if ( st > 0 ) {
        // check the response
        st = emaMgmtAT_CheckOK( context->response, &len );
        if ( st == MGMT_AT_ST_SUCCESS ) {
            // get the bid string
            loc = 0;
            msg = CheckForMsg( context->response, len, &loc );
            if ( msg ) {
                // validate "+OCBID:" string in msg
                if ( strstr( context->response, "+OCBID:" ) ) {
                    // null terminate for string ops, and update the output buff, skip "\r\n+OCBID: ", and quotes around BID string
                    *((char*)msg+len-2) = 0;
                    strcpy( formatting, (char*)msg+10 );
                }else{
                    st = MGMT_AT_ST_BAD_RESPONSE;
                }
            }
        }
    }
    return st;
}

/* 
emaModemAT_CheckOK: 
Checks an AT cmd response(*buff) for the OK pattern. The length of the data(*len) excluding the OK pattern is updated. 
*/
int emaMgmtAT_CheckOK( char *buff, unsigned short *len )
{
    int st = MGMT_AT_ST_SUCCESS;
    char *pLen;

    // check the input buff for the "\r\nOK\r\n" pattern
    pLen = strstr(buff, ema_atOKPattern); 
    if ( pLen == 0 ){
        st = MGMT_AT_ST_NO_OK;
    }else{
        *len = pLen - buff;
    }

    return st;
}

/* 
emaMgmtAT_DisableEcho: 
*/
int emaMgmtAT_DisableEcho( struct ema_mgmt_context *context )
{
    int st;
    unsigned short len;

    emaMgmtAT_BuildCmd(context, ema_atDisEchoCmd);

    // call the user provided hook for at comms
    st = (*ema_at_comm_handle)( context );
    if ( st > 0 ) {
        // check the response
        st = emaMgmtAT_CheckOK( context->response, &len );
    }
    return st;
}

/* 
emaModemAT_CheckOK: 
*/
int emaMgmtAT_EnableEcho( struct ema_mgmt_context *context )
{
    int st;
    unsigned short len;

    emaMgmtAT_BuildCmd(context, ema_atEnEchoCmd);

    // call the user provided hook for at comms
    st = (*ema_at_comm_handle)( context );
    if ( st > 0 ) {
        // check the response
        st = emaMgmtAT_CheckOK( context->response, &len );
    }
    return st;
}




/*###########################################*/
