/*
 * emaPlay.c
 *
 * Created: 6/24/2019 11:59:16 AM
 *  Author: matt.voigt
 */
 
#include "emaPlay.h"
#include "emaPlay_config.h"
#include "oc_leaf.h"

/*####### Timer Related Functionlaity #######*/
// these variables should NEVER be written to
uint32_t u32Sys1msTic;
uint32_t u32Sys1secTic;
uint32_t inlineTimer;  

static struct timer_task system_1ms_tick;
 
/* 
emaPlay_init_sys1ms_tick:
Initializes a timer that is used for a 1ms tick. The 1ms tick is then used by helper macros 
for creating/measuring delay when needed. 
*/
void emaPlay_init_sys1ms_tick( void )
{
    system_1ms_tick.interval = 1;
    system_1ms_tick.cb = (timer_cb_t)emaPlay_system_1ms_tick_cb;
    system_1ms_tick.mode = TIMER_TASK_REPEAT;
    timer_add_task(&TIMER_0, &system_1ms_tick);
    timer_start( &TIMER_0 );
}

/* 
emaPlay_system_1ms_tick_cb:
Increments the global tic variables that is used in the delay macros. 
*/
void emaPlay_system_1ms_tick_cb( void )
{
    static uint8_t sw0_d_cnt = 100;
    static uint8_t sw1_d_cnt = 100;
    static uint8_t sw0_pressed = 0;
    static uint8_t sw1_pressed = 0;
    static uint8_t status_pin_cnt = 50;

    uint8_t i;


    u32Sys1msTic++;
    if ( (u32Sys1msTic % 1000) == 0 ) {
        u32Sys1secTic++;
    }

    // scan switches
    if ( !gpio_get_pin_level( sw0_read ) ) {
        if ( sw0_d_cnt ) {
            sw0_d_cnt--;
        }else{
            // debounced pressed
            sw0_pressed = 1;
            sw0_d_cnt = 100;
        }
    }else{
        if ( sw0_pressed == 1 ) {
            // debounce released
            if ( sw0_d_cnt ) {
                sw0_d_cnt--;
            }else{
                // debounced press then released
                sw0_pending = 1;
                sw0_pressed = 0;
            }
        }else{
            sw0_d_cnt = 50;
        }
    }

    // scan switches
    if ( !gpio_get_pin_level(sw1_read) ) {
        if ( sw1_d_cnt ) {
            sw1_d_cnt--;
        }else{
            // debounced pressed
            sw1_pressed = 1;
            sw1_d_cnt = 100;
        }
    }else{
        if ( sw1_pressed == 1 ) {
            // debounce released
            if ( sw1_d_cnt ) {
                sw1_d_cnt--;
            }else{
                // debounced press then released
                sw1_pending = 1;
                sw1_pressed = 0;
            }
        }else{
            sw1_d_cnt = 50;
        }
    }

    // scan status pin
    if ( gpio_get_pin_level( ema_sts ) ) {
        // high
        if ( status_pin_cnt == 100 ) {
            sts_state = 1;
        }else{
            status_pin_cnt++;
        }
    }else{
        // low
        if ( status_pin_cnt ) {
            status_pin_cnt--;
        }else{
            sts_state = 0;
        }
    }

    // update leds
    for ( i=0; i < NUM_EMA_PLAY_LEDS; i++ ) {
        update_led( i );
    }

}
/*###########################################*/

/*####### Queue Related Functionlaity  ######*/
Q_STRUCT modem_uart_sync_rx_q, modem_uart_async_rx_q;
Q_STRUCT mgmt_uart_sync_rx_q, mgmt_uart_async_rx_q;
char modem_uart_sync_rx_q_buff[AT_RX_Q_BUFF_SIZE];
char modem_uart_async_rx_q_buff[AT_RX_Q_BUFF_SIZE];
char mgmt_uart_sync_rx_q_buff[EMA_AT_RX_Q_BUFF_SIZE];
char mgmt_uart_async_rx_q_buff[EMA_AT_RX_Q_BUFF_SIZE];

/* 
emaPlay_q_init:
Initializes a q that can be used for FIFO data storage. 
This function can also be called to reset the q to the initialized state. 
*/
void emaPlay_q_init( struct q_struct *q, char *buffer, unsigned buffer_size )
{
    setup_queue( q, buffer, buffer_size );
}

/* 
emaPlay_q_put:
Writes one byte(data) into an intialized q(*q). 
*/
int emaPlay_q_put( Q_STRUCT *q, char data )
{
    int q_st;

    q_st = char_into_queue(q, data);

    return q_st;
}

/* 
emaPlay_q_get:
Reads one byte(returns q_st) from an initilalized q(*q). Waits the specified timeout(timeout) 
before returning if the q(*q) is empty. 
*/
int emaPlay_q_get( Q_STRUCT *q, unsigned short timeout )
{
    int q_st;
    int timer = TIME_ELAPSED_ms(0);

    // wait for data to be available in the q
    do {
        q_st = char_outof_queue( q );
        // check timer expiry
        if ( TIME_ELAPSED_ms(timer) >= timeout ) {
            q_st = Q_ST_TIMEOUT;
            break;
        }
    } while ( q_st == Q_ST_EMPTY );

    return q_st;
}

/* 
emaPlay_q_check:
Checks a q(*q) for available data and returns the number of bytes(returns q_st) available. Waits a 
specified time(timeout) before checking the q. 
*/
int emaPlay_q_check( Q_STRUCT *q, unsigned short timeout )
{
    int q_st;

    // check for data after specified timeout
    if ( timeout ) {
        TIME_DELAY_ms(timeout);
    }
    q_st = num_chars_queued( q );

    return q_st;
}

/*###########################################*/

/*####### modem/ema mgmt UART Functionality ######*/
struct io_descriptor *ema_modem_uart_io;
struct io_descriptor *ema_mgmt_uart_io;
static uint8_t mod_uart_tx_complete, mgmt_uart_tx_complete;
static uint8_t mod_uart_sync_rx, mgmt_uart_sync_rx;

/* 
emaPlay_uart_setup: 
Completes the setup of an the ema:Play UART(*const uart). The initialization occurs during the atmel_start_init() call. 
This function registers the interrupt callbacks(tx/rx_callback), sets the read/write descriptor(**io_rw). 
*/
void emaPlay_uart_setup( struct usart_async_descriptor *const uart, struct io_descriptor **io_rw, usart_cb_t tx_callback, usart_cb_t rx_callback )
{
    // register the interrupt callbacks for the async uart driver
    usart_async_register_callback( uart, USART_ASYNC_TXC_CB, tx_callback );
    usart_async_register_callback( uart, USART_ASYNC_RXC_CB, rx_callback );

    // get the io descriptor for writes/reads for this uart
    usart_async_get_io_descriptor( uart, io_rw );
}

/* 
emaPlay_modem_tx_cb: 
This callback is executed after a successful transmission of the uart write buffer. Since 
the hardware handles the byte interrupts, only a flag is set here once the buffer transmit is 
complete. 
*/
void emaPlay_modem_tx_cb ( void )
{
    mod_uart_tx_complete = 1;
}

/* 
emaPlay_modem_rx_cb:
This callback is executed after 1 to 16 btes of data has been received by the UART. Data can be 
recieved synchronous to a transmit or Asynchronously(URC), independent of a transmit. 
*/
void emaPlay_modem_rx_cb ( void )
{
    uint8_t rx_data[16];
    uint8_t len;

    // data received, save it in the q 
    if ( mod_uart_sync_rx ) {
        // this is sync rx data
        if (mod_uart_tx_complete) {
            while ( usart_async_is_rx_not_empty(&ema_modem_uart) ) {
                len = io_read(ema_modem_uart_io, rx_data, 16);
                for ( uint8_t i=0; i<len; i++ ) {
                    emaPlay_q_put( &modem_uart_sync_rx_q, rx_data[i] );
                }
            }
        }
    }else{
        // this is async(URC) rx data, save in different buffer
        while ( usart_async_is_rx_not_empty(&ema_modem_uart) ) {
            len = io_read(ema_modem_uart_io, rx_data, 16);
            for ( uint8_t i=0; i<len; i++ ) {
                emaPlay_q_put( &modem_uart_async_rx_q, rx_data[i] );
            }
        }
    }
}

/* 
emaPlay_modem_uart_send: 
Sends x(len) bytes of data contained in the supplied buffer(*buffer). Waits a specfied time(timeout) for the 
transmission to complete. Returns a status code. 
*/
int emaPlay_modem_uart_send( uint8_t *buffer, uint16_t len, uint16_t timeout, uint8_t flags )
{
    int st = UART_ST_TX_SUCCESS;
    uint32_t timer;
    mod_uart_sync_rx = 0;

    #if DBG_MODEM_AT
    char str[256];
    int l;
    l = sprintf(str, "\r\n%d, AT CMD, Val=,\r\n", (int)u32Sys1msTic);
    emaPlay_dbg_out( str, l );
    emaPlay_dbg_out( (char*)buffer, len );
    emaPlay_dbg_out( "\r\n", 2 );
    #if DBG_MODEM_AT_HEX
    // print msg in hex format
    l = 0;
    if ( (len/2) > sizeof(str) ) {
        emaPlay_dbg_out( "**HEX to long**", 15 );
    }else{
        for (int i = 0; i < len; i++) {
            l += sprintf(str+l, "%.2X", buffer[i] );
        }
        emaPlay_dbg_out( str, l );
    }
    emaPlay_dbg_out( "\r\n", 2 );
    #endif
    #endif

    // clear the tx complete flag and start the tx
    mod_uart_tx_complete = 0;
    timer = TIME_ELAPSED_ms(0);    
    io_write(ema_modem_uart_io, buffer, len);

    // poll the tx complete flag until the tx is complete or timeout
    while ( mod_uart_tx_complete == 0 ) {
        if (TIME_ELAPSED_ms(timer) >= timeout) {
            st = UART_ST_TIMEOUT;
            emaPlay_dbg_out( "\r\ntx Timeout\r\n", 14 );
            break;
        }
    }

    // if a sync rx is expected, then let the rx interrupt know
    if ( mod_uart_tx_complete == 1 ) {
        if ( flags & MODULE_FL_SYNC_RX ) {
            mod_uart_sync_rx = 1;
        }
    }

    return st;
}

/* 
emaPlay_modem_uart_receive: 
Recieves x(*len) amount of bytes and stores them in a supplied output buffer(*buffer). The number(*len) 
of bytes received is updated accordingly. Waits a specified time(timeout) before retreiving the data. 
*/
int emaPlay_modem_uart_receive( uint8_t *buffer, uint16_t *len, uint16_t timeout, uint8_t flags, const char *pattern )
{
    static int st = UART_ST_TX_SUCCESS;
    int i;
    uint32_t timer;
    char *p;

    timer = TIME_ELAPSED_ms(0);
    i = 0;
    do {
        // check for timeout
        TIME_DELAY_ms(2);
        if ( TIME_ELAPSED_ms(timer) >= timeout ) {
            st = Q_ST_TIMEOUT;
            break;
        }
        // check for data
        st = emaPlay_q_check(&modem_uart_sync_rx_q, 0);
        if ( st > 0 ) {
            // data ready, retrieve it
            while ( st ) {
                *(buffer+i) = (uint8_t)emaPlay_q_get( &modem_uart_sync_rx_q, 5 );
                i++;
                st--;
            }
        }
        // check for desired pattern
        p = strstr( (char*)buffer, pattern );
        if ( p ) {
            // get remaining data if available
            TIME_DELAY_ms(20)
            st = emaPlay_q_check(&modem_uart_sync_rx_q, 0);
            if ( st > 0 ) {
                while ( st ) {
                    *(buffer+i) = (uint8_t)emaPlay_q_get( &modem_uart_sync_rx_q, 5 );
                    i++;
                    st--;
                }
            }
            st = UART_ST_TX_SUCCESS;
        }
    } while ( p == 0 );


    #if DBG_MODEM_AT
    if ( st == UART_ST_TX_SUCCESS ) {
        char str[256];
        int l;
        l = sprintf(str, "\r\n%d, AT RESPONSE, Val=,\r\n", (int)u32Sys1msTic);
        emaPlay_dbg_out( str, l );
        emaPlay_dbg_out( (char*)buffer, i );
        emaPlay_dbg_out( "\r\n", 2 );
        #if DBG_MODEM_AT_HEX
        // print msg in hex format
        l = 0;
        if ( (i/2) > sizeof(str) ) {
            emaPlay_dbg_out( "**HEX to long**", 15 );
        }else{
            for (int j = 0; j < i; j++) {
                l += sprintf(str+l, "%.2X", buffer[j] );
            }
            emaPlay_dbg_out( str, l );
        }
        emaPlay_dbg_out( "\r\n", 2 );
        #endif
    }else{
        emaPlay_dbg_out( "\r\nrx Timeout\r\n", 14 );
    }
    #endif

    // clear the sync rx flag
    mod_uart_sync_rx = 0;

    *len = i;
    return st;
}

/* 
void emaPlay_mgmt_tx_cb( void ) 
 
*/
void emaPlay_mgmt_tx_cb ( void )
{
    mgmt_uart_tx_complete = 1;
}

/* 
void emaPlay_mgmt_rx_cb( void ) 
 
*/
void emaPlay_mgmt_rx_cb ( void )
{
    uint8_t rx_data[16];
    uint8_t len;

    // data received, save it in the q 
    if ( mgmt_uart_sync_rx ) {
        // this is sync rx data
        if (mgmt_uart_tx_complete) {
            while ( usart_async_is_rx_not_empty(&ema_mgmt_uart) ) {
                len = io_read(ema_mgmt_uart_io, rx_data, 16);
                for ( uint8_t i=0; i<len; i++ ) {
                    emaPlay_q_put( &mgmt_uart_sync_rx_q, rx_data[i] );
                }
            }
        }
    }else{
        // this is async(URC) rx data, save in different buffer
        while ( usart_async_is_rx_not_empty(&ema_mgmt_uart) ) {
            len = io_read(ema_mgmt_uart_io, rx_data, 16);
            for ( uint8_t i=0; i<len; i++ ) {
                emaPlay_q_put( &mgmt_uart_async_rx_q, rx_data[i] );
            }
        }
    }
}

/* 
emaPlay_mgmt_uart_send: 
Sends x(len) bytes of data contained in the supplied buffer(*buffer). Waits a specfied time(timeout) for the 
transmission to complete. Returns a status code. 
*/
int emaPlay_mgmt_uart_send( uint8_t *buffer, uint16_t len, uint16_t timeout, uint8_t flags )
{
    int st = UART_ST_TX_SUCCESS;
    uint32_t timer;
    mgmt_uart_sync_rx = 0;

    // clear the tx complete flag and start the tx
    #if DBG_MGMT_AT
    char str[64];
    int l;
    l = sprintf(str, "\r\n%d, ema AT CMD, Val=,\r\n", (int)u32Sys1msTic);
    emaPlay_dbg_out( str, l );
    emaPlay_dbg_out( (char*)buffer, len );
    emaPlay_dbg_out( "\r\n", 2 );
    // print msg in hex format
    #if DBG_MGMT_AT_HEX
    l = 0;
    for ( int i=0; i<len; i++ ) {
        l += sprintf(str+l, "%.2X", buffer[i] );
    }
    emaPlay_dbg_out( str, l );
    emaPlay_dbg_out( "\r\n", 2 );
    #endif
    #endif

    // send the data
    mgmt_uart_tx_complete = 0;
    timer = TIME_ELAPSED_ms(0);    
    io_write(ema_mgmt_uart_io, buffer, len);

    // poll the tx complete flag until the tx is complete or timeout
    while ( mgmt_uart_tx_complete == 0 ) {
        if (TIME_ELAPSED_ms(timer) >= timeout) {
            st = UART_ST_TIMEOUT;
            break;
        }
    }

    // if a sync rx is expected, then let the rx interrupt know
    if ( mgmt_uart_tx_complete == 1 ) {
        if ( flags & MGMT_FL_SYNC_RX ) {
            mgmt_uart_sync_rx = 1;
        }
    }

    return st;
}

/* 
emaPlay_mgmt_uart_receive: 
Recieves x(*len) amount of bytes and stores them in a supplied output buffer(*buffer). The number(*len) 
of bytes received is updated accordingly. Waits a specified time(timeout) before retreiving the data. 
*/
int emaPlay_mgmt_uart_receive( uint8_t *buffer, uint16_t *len, uint16_t timeout )
{
    int st = UART_ST_TX_SUCCESS;
    int i;

    // check available data
    st = emaPlay_q_check( &mgmt_uart_sync_rx_q, timeout );
    if ( st > 0 ) {
        // data ready, retrieve it
        for ( i=0; i<st; i++ ) {
            *(buffer+i) = (uint8_t)emaPlay_q_get( &mgmt_uart_sync_rx_q, 5 );
        }
        #if DBG_MGMT_AT
        char str[256];
        int l;
        l = sprintf(str, "\r\n%d, ema AT RESPONSE, Val=,\r\n", (int)u32Sys1msTic);
        emaPlay_dbg_out( str, l );
        emaPlay_dbg_out( (char*)buffer, st );
        emaPlay_dbg_out( "\r\n", 2 );
        // print msg in hex format
        #if DBG_MGMT_AT_HEX
        l = 0;
        for ( i=0; i<st; i++ ) {
            l += sprintf(str+l, "%.2X", buffer[i] );
        }
        emaPlay_dbg_out( str, l );
        emaPlay_dbg_out( "\r\n", 2 );
        #endif
        #endif
    }
    // clear the sync rx flag
    mgmt_uart_sync_rx = 0;

    *len = st;
    return st;
}

/*###########################################*/

#ifdef DEBUG
/*####### emaPlay Debug Functionality ######*/

char http_msg[] = "POST /post HTTP/1.1\r\nHost: httpbin.org\r\nAccept: application/xhtml+xml\r\nConnection: keep-alive\r\nContent-Type: application/x-www-form-urlencoded; charset=utf-8\r\nContent-Length: 36\r\n\r\nKey1=abcd&Key2=123&Key3=Hello World!";
//char http_msg[] = "POST / HTTP/1.1\r\nHost: enprcdq0l7zg9.x.pipedream.net\r\nContent-Type: application/json\r\nContent-Length: 16\r\n\r\n{\"name\": \"Yoda\"}";
char http_rcv_buf[1500];

uint8_t exp_carrier;

struct io_descriptor *emaPlay_dbg_uart_io;
Q_STRUCT dbg_rx_q, dbg_tx_q;
char dbg_q_rx_buff[DBG_Q_BUFF_SIZE];
char dbg_q_tx_buff[DBG_Q_BUFF_SIZE];
static uint8_t dbg_uart_tx_complete;

/* 
void emaPlay_dbg_tx_cb( void ) 
 
*/
void emaPlay_dbg_tx_cb ( void )
{
    dbg_uart_tx_complete = 1;
}

/* 
void emaPlay_dbg_rx_cb( void ) 
 
*/
void emaPlay_dbg_rx_cb ( void )
{
    uint8_t rx_data[16];
    uint8_t len;

    while ( usart_async_is_rx_not_empty(&emaPlay_debug_uart) ) {
        len = io_read(emaPlay_dbg_uart_io, rx_data, 16);
        for ( uint8_t i=0; i<len; i++ ) {
            emaPlay_q_put( &dbg_rx_q, rx_data[i] );
        }
    }
}

/* 
void emaPlay_dbg_out( void )
 
*/
void emaPlay_dbg_out( char *dbg_data, uint16_t len )
{
    int i;

    // stuff the debug out data into the dbg tx q
    for ( i=0; i<len; i++ ) {
        emaPlay_q_put( &dbg_tx_q, dbg_data[i] );
    }
}

/* 
emaPlay_dbg_uart_send: 
*/
int emaPlay_dbg_uart_send( char *buffer, uint16_t len, uint16_t timeout, uint8_t flags )
{
    int st = UART_ST_TX_SUCCESS;
    uint32_t timer = TIME_ELAPSED_ms(0);

    // clear the tx complete flag and start the tx
    dbg_uart_tx_complete = 0;
    io_write(emaPlay_dbg_uart_io, (uint8_t*)buffer, len);

    // poll the tx complete flag until the tx is complete or timeout
    while ( dbg_uart_tx_complete == 0 ) {
        if (TIME_ELAPSED_ms(timer) >= timeout) {
            st = UART_ST_TIMEOUT;
            break;
        }
    }

    return st;
}

uint8_t halt_test = 1;
/* 
void dbg_task( void )
 
*/
char help0[] = "\r\n\r\n------ Cmd Menu ------\r\n";
char help1[] = "'q' = at+cesq\r\n'n' = network status\r\n'f' = ema FW version\r\n'o' = toggle on_off pin";
char help2[] = "\r\n'p' = toggle ema power\r\n'r' = reset module via pin\r\n'c' = set PDP context(AT&T/VZW)\r\n'h' = HTTP query";
char help3[] = "\r\n'a' = ema validation test(AT&T pri.)\r\n'v' = ema validation test(VZW pri.)\r\n's' = socket status\r\n";
void dbg_task ( struct ema_mgmt_context *mgmt_handle, struct modem_at_context *at_handle )
{
    int data, i, l;
    char tx_buff[DBG_Q_BUFF_SIZE];
    unsigned char temp[64];
    char str[64];
    static unsigned short len;
    uint8_t input;
    static unsigned char glPDPCntx = ATT_CONTEXT_ID;

    // send all available data
    while ( (data = emaPlay_q_check( &dbg_tx_q, 0 )) != 0 ) {
        for ( i=0; i<data; i++ ) {
            tx_buff[i] = emaPlay_q_get( &dbg_tx_q, 5 );
        }
        emaPlay_dbg_uart_send( tx_buff, data, 100, 0 );
    }

    // check for any input
    data = emaPlay_q_check( &dbg_rx_q, 0 );
    if ( data ) {
        // get the data and process it
        input = emaPlay_q_get( &dbg_rx_q, 5 );
        switch ( input ) {
        // test
        case '?': 
            emaPlay_dbg_uart_send( help0, sizeof(help0), 50, 0 );
            emaPlay_dbg_uart_send( help1, sizeof(help1), 50, 0 );
            emaPlay_dbg_uart_send( help2, sizeof(help2), 50, 0 );
            emaPlay_dbg_uart_send( help3, sizeof(help3), 50, 0 );
            break;
        case 'q':
            emaModemAT_GetSignalQuality( at_handle, &temp[0], &temp[1], &temp[2], &temp[3], &temp[4], &temp[5] ); 
            temp[6] = emaPlay_ema_qualify_signal( temp[4], temp[5] );
            l = sprintf( str, "\r\nema Signal Quality(0-4): %d\r\n", temp[6] );
            emaPlay_dbg_uart_send( str, l, 30, 0 );
            break;
        case 'n':
            emaModemAT_GetNetworkStatus( at_handle, &temp[0], &temp[1], (char*)&temp[2] );
            l = sprintf( str, "\r\nema Network Status(reg, acc tech, operator): %d, %d, %s\r\n", temp[0], temp[1], &temp[2] );
            emaPlay_dbg_uart_send( str, l, 50, 0 );
            break;
        case 'f': 
            emaMgmtAT_GetFWVersion( mgmt_handle, (char*)temp );
            l = sprintf( str, "\r\nema FW: %s\r\n", temp );
            emaPlay_dbg_uart_send( str, l, 30, 0 );
            break;
        case 'o':
            if ( gpio_get_pin_level( ema_on_off ) ) {
                emaPlay_dbg_uart_send("\r\n**MODULE OFF**\r\n", 17, 30, 0 );
            }else{
                emaPlay_dbg_uart_send("\r\n**MODULE ON**\r\n", 18, 30, 0 );
            }
            emaPlay_ema_on_off_control( EMA_ON_OFF_ST_TOGGLE );
            break;
        case 's':
            emaModemAT_SocketStatus( at_handle );
            break;
        case 'e':
            emaModemAT_EnableErrorReporting(at_handle, 2);
            break;
        case 'p':

            if ( gpio_get_pin_level( mod4V_dis ) ) {
                emaPlay_dbg_uart_send("\r\n**POWER APPLIED**\r\n", 21, 30, 0 );
            }else{
                emaPlay_dbg_uart_send("\r\n**POWER REMOVED**\r\n", 21, 30, 0 );
                mgmt_handle->urc_status = 0;
                // turn all leds off
                gpio_set_pin_level( blu_led0, true );
                gpio_set_pin_level( blu_led1, true );
                gpio_set_pin_level( blu_led2, true );
                gpio_set_pin_level( blu_led3, true );
                gpio_set_pin_level( red_led0, true );
                gpio_set_pin_level( red_led1, true );
            }

            emaPlay_ema_power_control( EMA_POWER_ST_TOGGLE );
            // enable the uart(s), after turning ema on to prevent junk recieved data
            usart_async_enable( &ema_modem_uart );
            usart_async_enable( &ema_mgmt_uart );
            break;
        case 'r':
            emaPlay_ema_reset( 1010 );
            break;
        case 'h':
            #if 0
            emaModemAT_DisableEcho( at_handle );
            //emaModemAT_HTTPQueryRaw( at_handle, 1, glPDPCntx, "enprcdq0l7zg9.x.pipedream.net", http_msg, http_rcv_buf, &len );
            /*emaModemAT_HTTPPost( at_handle, 0, 
                                            glPDPCntx, 
                                            "enprcdq0l7zg9.x.pipedream.net", 
                                            0, 
                                            "/", 
                                            "application/json",
                                            0,
                                            "{\"name\": \"Yoda\"}",
                                            sizeof("{\"name\": \"Yoda\"}")-1,
                                            http_rcv_buf,
                                            &http_rcv_buf[300],
                                            &len );*/
            emaModemAT_HTTPPost( at_handle, 0, 
                                            glPDPCntx, 
                                            "httpbin.org", 
                                            0, 
                                            "/post", 
                                            "application/x-www-form-urlencoded; charset=utf-8",
                                            0,
                                            "Key1=abcd&Key2=123&Key3=Hello World!",
                                            36,
                                            http_rcv_buf,
                                            &http_rcv_buf[300],
                                            &len );
            #endif

            break;
        case 'c':
            if (glPDPCntx == ATT_CONTEXT_ID) {
                glPDPCntx = VZW_CONTEXT_ID;
                emaPlay_dbg_uart_send("\r\n**PDP Cntx: VZW**\r\n", 21, 30, 0 );
            }else{
                glPDPCntx = ATT_CONTEXT_ID;
                emaPlay_dbg_uart_send("\r\n**PDP Cntx: AT&T**\r\n", 22, 30, 0 );
            }
            break;
        case 'a':
            // start ema test
            if ( halt_test == 0 ) {
                halt_test = 1;
                emaPlay_dbg_uart_send("\r\nStopping Test....\r\n", sizeof("\r\nStopping Test....\r\n"), 30, 0 );
            }else{
                halt_test = 0;
                exp_carrier = 0; // att
                emaPlay_dbg_uart_send("\r\nStarting Test, AT&T ema\r\n", sizeof("\r\nStarting Test....\r\n"), 30, 0 );
            }
            break;
        case 'v':
            // start ema test
            if ( halt_test == 0 ) {
                halt_test = 1;
                emaPlay_dbg_uart_send("\r\nStopping Test....\r\n", sizeof("\r\nStopping Test....\r\n"), 30, 0 );
            }else{
                halt_test = 0;
                exp_carrier = 1; // vzw
                emaPlay_dbg_uart_send("\r\nStarting Test, VZW ema\r\n", sizeof("\r\nStarting Test, VZW ema\r\n"), 30, 0 );
            }
            break;
        default:
            break;
        }
    }
}

#endif

/*###########################################*/

/*##########  AT cmd Functionality ##########*/
/* 
emaPlay_modem_at_query:
Sends and receives an AT transaction/query. The tx and rx buffers are passed in as part of the at_query argument. 
This function clears the receive buffer, calucaltes the length of the AT cmd, updates any flags, sends the AT 
cmd, and waits for a response. A status code is returned. 
*/
int emaPlay_modem_at_query( struct modem_at_context *at_query, unsigned short tx_timeout, unsigned short res_timeout, const char *res_pattern )
{
    int at_st;
    uint16_t i;

    // flush the rx buffer
    memset( at_query->response, 0, sizeof(at_query->response) );

    // figure out the length of the at cmd/data
    if ( at_query->data_length == 0 ) {
        if (at_query->at_cmd_flags & MODULE_FL_SPECIAL_CMD) {
            at_query->at_cmd_flags &= ~MODULE_FL_SPECIAL_CMD;
            for (i = 0;; ++i) {
                if (*(at_query->at_cmd+i) == SEND_DATA_CONFIRM ) {
                    break;
                }
            }
        }else{
            for (i = 0;; ++i) {
                if (*(at_query->at_cmd+i) == '\r' ) {
                    break;
                }
            }
        }
        i++;
    }else{
        i = at_query->data_length;
    }

    // a sync rx(response) is always expected for a query
    at_query->at_cmd_flags |= MODULE_FL_SYNC_RX;

    // send/recieve the at query
    at_st = emaPlay_modem_uart_send( (uint8_t*)at_query->at_cmd, i, tx_timeout, at_query->at_cmd_flags );
    if (at_st == UART_ST_TX_SUCCESS) {
        at_st = emaPlay_modem_uart_receive((uint8_t *)at_query->response, &at_query->response_length, res_timeout, at_query->response_flags, res_pattern );
    }

    // clear all flags
    at_query->at_cmd_flags = 0;
    at_query->response_flags = 0;
    at_query->data_length = 0;

    return at_st;
}

int emaPlay_modem_async_recv( struct modem_at_context *at_query, unsigned short wait_time )
{
    int st, i;

    // poll the asynch rx buffer
    st = emaPlay_q_check( &modem_uart_async_rx_q, wait_time );
    if ( st > 0 ) {
        // data ready, retrieve it
        for ( i=0; i<st; i++ ) {
            *(at_query->async_response + i) = (uint8_t)emaPlay_q_get(&modem_uart_async_rx_q, 5);
        }
        #ifdef DEBUG
        char str[64];
        int l;
        l = sprintf(str, "\r\n%d, ASYNCH RESPONSE, Val=,\r\n", (int)u32Sys1msTic);
        emaPlay_dbg_out( str, l );
        emaPlay_dbg_out( (char*)at_query->async_response, st );
        emaPlay_dbg_out( "\r\n", 2 );
        // print msg in hex format
        l = 0;
        for ( int i=0; i<st; i++ ) {
            l += sprintf(str+l, "%.2X", at_query->async_response[i] );
        }
        emaPlay_dbg_out( str, l );
        emaPlay_dbg_out( "\r\n", 2 );
        #endif
    }

    // clear the sync rx flag
    mod_uart_sync_rx = 0;

    at_query->response_length = st;
    return st;

}

/* 
emaPlay_mgmt_at_query:
Sends and receives an AT transaction/query. The tx and rx buffers are passed in as part of the at_query argument. 
This function clears the receive buffer, calucaltes the length of the AT cmd, updates any flags, sends the AT 
cmd, and waits for a response. A status code is returned. 
*/
int emaPlay_mgmt_at_query( struct ema_mgmt_context *at_query )
{
    int at_st;
    uint16_t i;

    // flush the rx buffer
    memset( at_query->response, 0, sizeof(at_query->response) );

    // figure out the length of the at cmd;
    for ( i=0;;++i ) {
        if (*(at_query->at_cmd+i) == '\r' ) {
            break;
        }
    }
    i++;

    // a sync rx(response) is always expected for a query
    at_query->at_cmd_flags |= MGMT_FL_SYNC_RX;

    // send/recieve the at query
    at_st = emaPlay_mgmt_uart_send( (uint8_t*)at_query->at_cmd, i, 50, at_query->at_cmd_flags );
    if (at_st == UART_ST_TX_SUCCESS) {
        at_st = emaPlay_mgmt_uart_receive( (uint8_t*)at_query->response, &at_query->response_length, 250 );
    }

    return at_st;
}

extern int CheckForMsg( char *buff, unsigned short len, int *loc );
/* 
emaPlay_mgmt_URC_handler:
Processes any Asynchronous msgs/URCs that have been received from ema. This function must be 
called frequently periodic. Also, the user can insert code into this function to process the URCs. Also, the input 
parameters can be modified as needed. 
*/
void emaPlay_mgmt_URC_handler( struct ema_mgmt_context *handle )
{
    int st, urc, msg;
    static char urc_buff[EMA_AT_CMD_RESPONSE_MAX_SIZE];
    int loc = 0;
    static int len = 0;

    // check to see if any async data is available
    st = emaPlay_q_check( &mgmt_uart_async_rx_q, 0 );
    if ( st > 0 ) {
        // we have async data, pull data out of the q until we've received a full msg, indicated by 2 pairs of "\r\n"
        msg = 0;
        loc = 0;
        while ( msg == 0 && st ) {
            urc_buff[len] = emaPlay_q_get( &mgmt_uart_async_rx_q, 5 );
            len++;
            st--;
            msg = CheckForMsg( urc_buff, len, &loc );
        }

        // process a the msg if found
        if ( msg ) {
            urc = emaMgmtURC_GetValue(urc_buff, len);
            if ( urc != MGMT_URC_INVALID ) {

                #if DBG_MGMT_URC
                char str[256];
                int l;
                l = sprintf(str, "\r\n%d, ema URC, Val=%d, ", (int)u32Sys1msTic, urc);
                emaPlay_dbg_out( str, l );
                emaPlay_dbg_out( urc_buff, len );
                emaPlay_dbg_out( "\r\n", 2 );
                #if DBG_MGMT_URC_HEX
                // print msg in hex format
                l = 0;
                for ( int i=0; i<len; i++ ) {
                    l += sprintf(str+l, "%.2X", urc_buff[i] );
                }
                emaPlay_dbg_out( str, l );
                emaPlay_dbg_out( "\r\n", 2 );
                #endif
                #endif

                // we have a urc process them according to the user application
                emaPlay_mgmt_process_URC( handle, urc );
            }
            len = 0;
        }
    }
}

void emaPlay_mgmt_process_URC( struct ema_mgmt_context *handle, int urc )
{
    // process the urc according to the application & hardware
    switch ( urc ) {
    case MGMT_URC_AT_READY:
        handle->urc_status |= MGMT_CNTX_EMA_AT_READY;
        break;
    case MGMT_URC_POWERING_ON:
        handle->urc_status |= MGMT_CNTX_MODULE_POWERING_ON;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERING_OFF;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERED_OFF;
        break;
    case MGMT_URC_READY:
        handle->urc_status |= MGMT_CNTX_MODULE_READY;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERING_OFF;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERED_OFF;
        break;
    case MGMT_URC_POWERING_OFF:
        handle->urc_status |= MGMT_CNTX_MODULE_POWERING_OFF;
        handle->urc_status &= ~MGMT_CNTX_MODULE_READY;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERING_ON;
        break;
    case MGMT_URC_POWERED_OFF:
        handle->urc_status |= MGMT_CNTX_MODULE_POWERED_OFF;
        handle->urc_status &= ~MGMT_CNTX_MODULE_READY;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERING_ON;
        break;
    case MGMT_URC_MODULE_UART_ESTABLISHED:
        handle->urc_status |= MGMT_CNTX_MODULE_UART_READY;
        handle->urc_status &= ~MGMT_CNTX_MODULE_UART_DOWN;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERING_OFF;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERED_OFF;
        break;
    case MGMT_URC_MODULE_UART_REMOVED:
        handle->urc_status |= MGMT_CNTX_MODULE_UART_DOWN;
        handle->urc_status &= ~MGMT_CNTX_MODULE_UART_READY;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERED_OFF;
        break;
    case MGMT_URC_MODULE_RESET_REQUESTED:
        handle->urc_status |= MGMT_CNTX_MODULE_RESET_REQ;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERING_OFF;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERED_OFF;
        break;
    case MGMT_URC_MODULE_RESETTING:
        handle->urc_status |= MGMT_CNTX_MODULE_RESETTING;
        handle->urc_status &= ~MGMT_CNTX_MODULE_RESET_SHORT;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERING_OFF;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERED_OFF;
        break;
    case MGMT_URC_MODULE_RESET_SUCCESS:
        handle->urc_status |= MGMT_CNTX_MODULE_RESET;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERING_OFF;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERED_OFF;
        break;
    case MGMT_URC_MODULE_RESET_NOT_OBSERVED:
        handle->urc_status |= MGMT_CNTX_MODULE_RESET_SHORT;
        handle->urc_status &= ~MGMT_CNTX_MODULE_RESET_REQ;
        handle->urc_status &= ~MGMT_CNTX_MODULE_RESETTING;
        handle->urc_status &= ~MGMT_CNTX_MODULE_RESET;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERING_OFF;
        handle->urc_status &= ~MGMT_CNTX_MODULE_POWERED_OFF;
        break;
    default:
        break;
    }

}

uint8_t emaPlay_waitURC( struct ema_mgmt_context *mgmt_handle, uint32_t URCMask, uint16_t wait_time )
{
    uint8_t ret = 0;
    uint32_t timer = TIME_ELAPSED_ms(0);

    do {
        // check timeout
        if ( TIME_ELAPSED_ms(timer) >= wait_time ) {
            // timeout
            break;
        }
        // check the urc status against the mask 
        emaPlay_mgmt_URC_handler( mgmt_handle );
        if ( (mgmt_handle->urc_status & URCMask) == URCMask ) {
            ret = 1;
        }
    } while (ret == 0);

    return ret;
}

uint8_t emaPlay_checkURC( struct ema_mgmt_context *mgmt_handle, uint32_t URCMask )
{
    uint8_t ret = 0;

    if ( (mgmt_handle->urc_status & URCMask) == URCMask ) {
        ret = 1;
    }

    return ret;
}

/*###########################################*/



/*###### ema:Play GPIO Functionality  #######*/

/* 
emaPLay_ema_on_off_control:
*/
void emaPlay_ema_on_off_control( enum ema_on_off_state state )
{
    switch ( state ) {
    case EMA_ON_OFF_ST_ON:
        gpio_set_pin_level( ema_on_off, true );
        break;
    case EMA_ON_OFF_ST_OFF:
        gpio_set_pin_level( ema_on_off, false );
        break;
    case EMA_ON_OFF_ST_TOGGLE:
        gpio_toggle_pin_level( ema_on_off );
        break;
    default:
        gpio_set_pin_level( ema_on_off, false );
        break;
    }
}

/* 
emaPLay_ema_on_off_control:
*/
void emaPlay_ema_power_control( enum ema_power_state state )
{
    switch ( state ) {
    case EMA_POWER_ST_ON:
        gpio_set_pin_level( mod4V_dis, false );
        usart_async_enable( &ema_modem_uart );
        usart_async_enable( &ema_mgmt_uart );
        break;
    case EMA_POWER_ST_OFF:
        gpio_set_pin_level( mod4V_dis, true );
        usart_async_disable( &ema_modem_uart );
        usart_async_disable( &ema_mgmt_uart );
        break;
    case EMA_POWER_ST_TOGGLE:
        gpio_toggle_pin_level( mod4V_dis );
        break;
    default:
        gpio_set_pin_level( mod4V_dis, false );
        break;
    }
}

void emaPlay_ema_reset( uint16_t reset_time )
{
    // reset ema using the reset pin
    gpio_set_pin_level(ema_reset, true);
    TIME_DELAY_ms( reset_time );
    gpio_set_pin_level(ema_reset, false);
}
/*###########################################*/

/*###### ema:Play App level functions  #######*/
/* 
emaPLay_ema_on_off_control:
*/
uint8_t sw0_pending, sw1_pending;
uint8_t sts_state;

uint8_t emaPlay_check_events( struct emaPlay_status *emaPlay )
{
    // locals vars
    static uint8_t ev[NUM_EMA_PLAY_EV];
    int diff;

    // get the latest state of events
    if ( sw0_pending ) {
        sw0_pending = 0;
        ev[EV_SW0_PRESS_RELEASE] = 1;
    }else{
        ev[EV_SW0_PRESS_RELEASE] = 0;
    }
    if ( sw1_pending ) {
        sw1_pending = 0;
        ev[EV_SW1_PRESS_RELEASE] = 1;
    }else{
        ev[EV_SW1_PRESS_RELEASE] = 0;
    }

    // compare to current emaPlay status
    diff = memcmp( emaPlay->events, ev, 4 );
    if ( diff ) {
        // update emaPlay status
        memcpy( emaPlay->events, ev, 4 );
		return 1;
    }else{
        return 0;
    }
}

void emaPlay_leaf_cmd_handler(char *command, char **params, int num_params) {
    // led cmds
    if (!strcmp(command, "service_control")) {
        if (num_params > 1) {
            if (!strcmp(params[0], "Stop")) {
                //set_led(params[1], true);
            } else if (!strcmp(params[0], "Start")) {
                //set_led(params[1], false);
            }
        }
    }
    // buzzer cmds
    if (!strcmp(command, "buzzer")) {
        if (num_params == 1) {
            if ( !strcmp(params[0], "on") ) {
                //alertCmd = 1;
            } else if ( !strcmp(params[0], "off") ) {
                //alertCmd = 0;
            }
        }
    }
}

uint8_t timed_out( uint16_t timer, uint32_t time_to_err )
{
    uint8_t err = 0;

    if ( (timer * SYS_TASK_EXECUTION_INTERVAL) >= time_to_err ) {
        err = 1;
    }

    return err;
}


void emaPlay_read_temp_humidiity( struct emaPlay_status *emaPlay )
{
	uint8_t tx_buff[2];
	uint8_t rx_buff[6];
    uint16_t dataRaw;
    float tempF, rHum;

    tx_buff[0] = 0x24;  // no clock stretching
    tx_buff[1] = 0x00;  // high repeatability
    i2c_m_sync_cmd_write( &I2C, 0x45, tx_buff, 2 );

    //TIME_DELAY_ms( 2 );

    i2c_m_sync_cmd_read( &I2C, 0x45, rx_buff, 6 );

    // temp conversion formula
    // Tf = -49 + 315 x (St(dec) / 2^16 - 1), where St is bytes 0-msb, 1-lsb of rx_buff
    dataRaw = rx_buff[0] << 8;
    dataRaw |= rx_buff[1];
    tempF = -49 + 315 * ((float)dataRaw / 65535);

    // humidity conversion formula
    // Rh = 100 * ( Srh(dec) / 2^16 - 1) 
    dataRaw = rx_buff[3] << 8;
    dataRaw |= rx_buff[4];
    rHum = 100 * ( (float)dataRaw / 65535 );

    emaPlay->temp = tempF - TEMP_OFFSET;
    emaPlay->r_hum = rHum;
}

void emaPlay_scan_analog_inputs( struct emaPlay_status *emaPlay )
{
    static uint8_t adc_cnt = 0;
    static uint16_t adc_val[MAX_NUM_ADC_INPUTS];
    uint16_t cur_val;
    uint8_t i, adc_pin;

    // turn on the photo detector
    gpio_set_pin_level( n_photosense_en, false );

    // average the adc values for each channel
    if ( adc_cnt < NUM_ADC_READS_TO_AVG ) {
        adc_cnt++;
        adc_pin = ADC_INPUTCTRL_MUXPOS_AIN12;
        for ( i=0; i<MAX_NUM_ADC_INPUTS; i++ ) {
            // set the pos & neg channels
            adc_sync_set_inputs( &ADC_0, adc_pin+i, 0x18/*gnd*/, 0 );
            // read the value
            adc_sync_read_channel( &ADC_0, 0, (uint8_t*)&cur_val, 2 );
            // add to last reading 
            adc_val[i] += cur_val;
        }
    }else{
        // avg the readings
        emaPlay->light = adc_val[0] / NUM_ADC_READS_TO_AVG;
        emaPlay->ema_v = adc_val[1] / NUM_ADC_READS_TO_AVG; 
        emaPlay->sys_v = adc_val[2] / NUM_ADC_READS_TO_AVG; 
        emaPlay->user_adc = adc_val[3] / NUM_ADC_READS_TO_AVG;
        // reset vars
        for ( i=0; i<NUM_ADC_READS_TO_AVG; i++ ) {
            adc_val[i] = 0;
        }
        adc_cnt = 0;
    }

    // turn off the photo detector
    gpio_set_pin_level( n_photosense_en, true );
}

static char leaf_msg_data[512];
static const char event_json_names[NUM_EMA_PLAY_EV][EV_JSON_NAME_SIZE] = 
{
    {"sw1_state"},
    {"sw2_state"},
    {"din1_state"},
    {"din2_state"}
};

uint16_t emaPlay_populate_leaf_event_data( char *leaf_data_buff, uint8_t *event_data )
{
    uint8_t i;
    int l;

    // grab the latest event data and build the leaf json msg from it
    l = sprintf( leaf_data_buff, "{\"type\": \"ema/Canopy Demo\",\"events\":{" );
    for ( i=0; i<NUM_EMA_PLAY_EV; i++ ) {
        l += sprintf( leaf_data_buff+l, "\"%s\": %d,", (char*)&event_json_names[i][0], (int)event_data[i] );
    }
    l += sprintf( leaf_data_buff+l, "}}" );

    return (uint16_t)l;
}

void emaPlay_system_task(struct ema_mgmt_context *mgmt_handle, 
                         struct modem_at_context *modem_handle, 
                         struct ema_status *ema, 
                         struct emaPlay_status *emaPlay,  
                         uint8_t *ev )
{
    static uint8_t system_state = SYS_ST_INIT_EMA_APPLY_POWER;
    static uint8_t err_code;
    uint8_t urc_st = 0;
    int at_st = 0;
    int st = 0;
    static uint16_t delay = 0;          
    static uint16_t error_timer = 0;
    static uint8_t event_data[NUM_EMA_PLAY_EV];
    static uint32_t stat_msg_timer;
        #define  SYS_STAT_MSG_INTERVAL   60000 // 60 seconds
    static uint32_t noop_msg_timer;
        #define  SYS_NOOP_MSG_TIMER      30000 // 30 seconds
    static enum LeafMessageTypes leaf_msg_type;
    uint16_t data_len;
    static int leaf_msg_len = 0;
    static int leaf_retry = LEAF_ATTEMPTS;

    if ( delay ) {
        delay--;
    }else{
        switch (system_state) {
        case SYS_ST_NORMAL_OP:
            // normal operation priorities:
            // 1. check for events occured
            //      a. dynamic emaPLay/leaf events
            //      b. Periodic leaf stat msg
            //      c. Periodic Leaf noop msg to get cmds
            // 2. check network status

            // handle an event, that needs to be sent using leaf
            if ( *ev ) {
                // clear the event
                *ev = 0;

                // snapshot the event data and build the leaf msg
                memcpy( event_data, emaPlay->events, NUM_EMA_PLAY_EV );
                leaf_msg_type = LEAF_EVENT;
                system_state = SYS_ST_BUILD_LEAF_MSG;
                break;
            }

            // periodic stat msg
            if ( TIME_ELAPSED_ms(stat_msg_timer) >= SYS_STAT_MSG_INTERVAL ) {
                stat_msg_timer = TIME_ELAPSED_ms( 0 );
                leaf_msg_type = LEAF_STATISTIC;
                system_state = SYS_ST_BUILD_LEAF_MSG;
                break;
            }

            #if ENABLE_LEAF_NOOP
            // periodic noop msg
            if ( TIME_ELAPSED_ms(noop_msg_timer) >= SYS_NOOP_MSG_TIMER) {
                //noop_msg_timer = TIME_ELAPSED_ms( 0 );
                leaf_msg_type = LEAF_NOOP;
                system_state = SYS_ST_BUILD_LEAF_MSG;
            }
            #endif

            break;
        case SYS_ST_INIT_EMA_APPLY_POWER:
            // apply power to ema, module is still off

            emaPlay_ema_power_control( EMA_POWER_ST_ON );
            // wait for the power on URC
            urc_st = emaPlay_checkURC( mgmt_handle, MGMT_CNTX_EMA_AT_READY );
            if ( urc_st ) {
                // check ema AT comms
                at_st = emaMgmtAT_CheckComm( mgmt_handle );
                if ( at_st == MGMT_AT_ST_SUCCESS ) {
                    // comms good, disable echo, get the ema sn and fw ver
                    emaMgmtAT_DisableEcho( mgmt_handle );
                    emaMgmtAT_GetSN( mgmt_handle, ema->sn );
                    emaMgmtAT_GetFWVersion( mgmt_handle, ema->fw_ver );
                    // move to next state
                    system_state = SYS_ST_INIT_EMA_TURN_ON_MODULE;
                    error_timer = 0;
                }else{
                    // error
                    system_state = SYS_ST_INIT_ERROR;
                    err_code = ERR_CODE_EMA_AT_RESPONSE;
                }
            }else{
                if ( timed_out( error_timer, EMA_APPLY_POWER_WAIT_TIME ) ) {
                    system_state = SYS_ST_INIT_ERROR;
                    err_code = ERR_CODE_URC_TIME_OUT;
                    break;
                }
                // delay 500ms sec then check again
                delay = 500 / SYS_TASK_EXECUTION_INTERVAL;
                // track how long its taking to get the urc
                error_timer += delay;
            }

            break;
        case SYS_ST_INIT_EMA_TURN_ON_MODULE:
            // ema has power, turn module on

            emaPlay_ema_on_off_control( EMA_ON_OFF_ST_ON );
            // wait for the module and interface ready URCs
            urc_st = emaPlay_checkURC( mgmt_handle, MGMT_CNTX_MODULE_POWERING_ON | MGMT_CNTX_MODULE_READY | MGMT_CNTX_MODULE_UART_READY );
            if ( urc_st ) {
                // on-board module is ready, check comms
                at_st = emaModemAT_CheckComm( modem_handle );
                if ( at_st == MOD_AT_ST_SUCCESS ) {
                    // comms good, disable echo
                    emaModemAT_DisableEcho( modem_handle );
                    system_state = SYS_ST_CHECK_NETWORK;
                    error_timer = 0;
                }else{
                    // error
                    system_state = SYS_ST_INIT_ERROR;
                    err_code = ERR_CODE_EMA_MODULE_RESPONSE;
                }
            }else{
                if ( timed_out( error_timer, EMA_MODULE_READY_WAIT_TIME ) ) {
                    system_state = SYS_ST_INIT_ERROR;
                    err_code = ERR_CODE_URC_TIME_OUT;
                    break;
                }
                // delay =500 msec then check again
                delay = 500 / SYS_TASK_EXECUTION_INTERVAL;
                // track how long its taking to get the urc
                error_timer += delay;
            }

            break;
        case SYS_ST_CHECK_NETWORK:
            // checks emas network status, before moving on
            
            at_st = (uint8_t)emaPlay_check_network( modem_handle, ema );
            if ( at_st == 1 ) {
                system_state = SYS_ST_INIT_LEAF;
                error_timer = 0;
            }else{
                if ( timed_out( error_timer, EMA_NETWORK_READY_TIME ) ) {
                    system_state = SYS_ST_INIT_ERROR;
                    err_code = ERR_CODE_NETWORK_ERROR;
                    break;
                }
                // delay 5 sec then check again
                delay = 5000 / SYS_TASK_EXECUTION_INTERVAL;
                // track how long its taking to get the urc
                error_timer += delay;
            }

            break;
        case SYS_ST_INIT_LEAF:
            // init leaf agent

            stat_msg_timer = TIME_ELAPSED_ms( 0 );
            noop_msg_timer = TIME_ELAPSED_ms( 0 );

            st = start_leaf( modem_handle, ema->cntx_id, ema->sn, "leaf-uat.gocanopy.io", "banyan", "leaf@oc.beta", "l3@fBeta", emaPlay_leaf_cmd_handler );
            if ( st == 1 ) {
                // the leaf agent on ema:Play is now ready, go wait for something to cause a leaf msg
                system_state = SYS_ST_NORMAL_OP;
            }

            break;
        case SYS_ST_BUILD_LEAF_MSG:
            // builds a leaf msg
            
            // grab the relevant ema:Play data
            if ( leaf_msg_type == LEAF_EVENT ) {
                data_len = emaPlay_populate_leaf_event_data( leaf_msg_data, event_data );
            } else if ( leaf_msg_type == LEAF_STATISTIC ) {

            } 

            if ( leaf_msg_type == LEAF_NOOP ) {
                leaf_msg_len = build_leaf_noop( );
            }else{
                leaf_msg_len = build_leaf_message(leaf_msg_type, "", leaf_msg_data);
            }

            // send the msg
            system_state = SYS_ST_SEND_LEAF_MSG;

            break;
        case SYS_ST_SEND_LEAF_MSG:
            // send the leaf msg to canopy

            // send the leaf event/stat msg with retries
            if ( leaf_retry ) {
                st = send_leaf_query(modem_handle, ema->cntx_id, leaf_msg_len);
                if ( st == 0 ) {
                    // leaf msg fail, remaing in this state
                    leaf_retry--;
                }else{
                    // leaf msg success
                    noop_msg_timer = TIME_ELAPSED_ms( 0 );
                    system_state = SYS_ST_NORMAL_OP;
                }
            }else{
                // exhausted retries
                leaf_retry = LEAF_ATTEMPTS;
                system_state = SYS_ST_NORMAL_OP;
            }

            break;
        case SYS_ST_PROCESS_LEAF_CMD:
            // handle a cmd coming from canopy

            break;
        case SYS_ST_INIT_ERROR:
            // set hw for error condition

            emaPlay_set_led_array( LED_PATTERN_PAIR_ERROR, 
                                   LED_PATTERN_PAIR_ERROR,
                                   LED_PATTERN_PAIR_ERROR,
                                   LED_PATTERN_PAIR_ERROR );

            system_state = SYS_ST_ERROR;

            // fall through
        case SYS_ST_ERROR:
            // let the user know an error occured with the leds

            // process error code
            switch (err_code) {
            case ERR_CODE_URC_TIME_OUT:

                break;
            case ERR_CODE_EMA_AT_RESPONSE:

                break;
            case ERR_CODE_EMA_MODULE_RESPONSE:

                break;
            case ERR_CODE_LEAF_RESPONSE:

                break;
            default:
                break;
            }

            break;
        default:
            break;
        }
    }

}

uint8_t emaPlay_check_network( struct modem_at_context *modem_handle, struct ema_status *ema )
{
    int at_st;
    uint8_t net_st = 0;

    at_st = emaModemAT_GetNetworkStatus(modem_handle, &ema->reg, &ema->acc_tech, ema->carrier);
    if ( ema->reg == 1 ) {
        // check signal level
        at_st = emaModemAT_GetSignalQuality( modem_handle, &ema->rxlev, &ema->ber, &ema->rscp, &ema->ecno, &ema->rsrq, &ema->rsrp );
        if ( at_st == MOD_AT_ST_SUCCESS ) {
            ema->calc_sig_q = emaPlay_ema_qualify_signal( ema->rsrq, ema->rsrp );
            emaPlay_update_led_graph( &ema->calc_sig_q );
            if ( ema->calc_sig_q != SIGNAL_UNKNOWN ) {
                // registered and acquired signal qual
                // update the carrier info and cntx id
                if ( strstr( ema->carrier, "AT&T" ) ) {
                    ema->cntx_id = ATT_CONTEXT_ID;
                } else if ( strstr( ema->carrier, "Verizon" ) ) {
                    ema->cntx_id = VZW_CONTEXT_ID;
                }
                net_st = 1;
            }
        }
    }else{
        emaPlay_set_led_array( LED_PATTERN_PAIR_SEARCHING, 
                               LED_PATTERN_PAIR_OFF,
                               LED_PATTERN_PAIR_OFF,
                               LED_PATTERN_PAIR_OFF );
    }

    return net_st;
}

/* 
emaPLay_ema_apply_power:
*/
void emaPlay_ema_apply_power( void )
{
    emaPlay_ema_power_control( EMA_POWER_ST_ON );
}

/* 
emaPLay_ema_remove_power:
*/
void emaPlay_ema_remove_power( void )
{
    emaPlay_ema_power_control( EMA_POWER_ST_OFF );
}

/* 
emaPLay_ema_turn_on:
*/
void emaPlay_ema_turn_on( void )
{
    emaPlay_ema_on_off_control( EMA_ON_OFF_ST_ON );
}

/* 
emaPLay_ema_turn_off:
*/
void emaPlay_ema_turn_off( void )
{
    emaPlay_ema_on_off_control( EMA_ON_OFF_ST_OFF );
}

/* 
emaPlay_ema_qualify_signal:
*/
unsigned char emaPlay_ema_qualify_signal( unsigned char rsrq, unsigned char rsrp )
{
    unsigned char s_q, s_q1, s_q2;

    // place the values into a range and qualify them
    if ( rsrq == 255 || rsrq == 0 ) {
        s_q1 = SIGNAL_UNKNOWN;
    } else if (rsrq >= 20) {
        s_q1 = SIGNAL_EXCELLENT;
    } else if ( rsrq >= 10 ) {
        s_q1 = SIGNAL_GOOD;
    } else if ( rsrq >= 1 ) {
        s_q1 = SIGNAL_FAIR;
    } else{
        s_q1 = SIGNAL_POOR;
    }

    // place the values into a range and qualify them
    if ( rsrp == 255 || rsrp == 0 ) {
        s_q2 = SIGNAL_UNKNOWN;
    } else if (rsrp >= 60) {
        s_q2 = SIGNAL_EXCELLENT;
    } else if ( rsrp >= 51 ) {
        s_q2 = SIGNAL_GOOD;
    } else if ( rsrp >= 41 ) {
        s_q2 = SIGNAL_FAIR;
    } else{
        s_q2 = SIGNAL_POOR;
    }

    // take the lower of the two
    if ( s_q1 == s_q2 ) {
        s_q = s_q1;
    } else if ( s_q1 < s_q2 ) {
        s_q = s_q1;
    } else{
        s_q = s_q2;
    }

    return s_q;
}

void emaPlay_set_led_array( uint16_t duty_cycle0, uint16_t period0,
                            uint16_t duty_cycle1, uint16_t period1,
                            uint16_t duty_cycle2, uint16_t period2,
                            uint16_t duty_cycle3, uint16_t period3 )
{   
    // update the entire led array
    set_led_blink_pattern( 0, duty_cycle0, period0 );
    set_led_blink_pattern( 1, duty_cycle1, period1 );
    set_led_blink_pattern( 2, duty_cycle2, period2 );
    set_led_blink_pattern( 3, duty_cycle3, period3 );
}

void emaPlay_update_led_graph( uint8_t *sig )
{
    if ( *sig == SIGNAL_EXCELLENT ) {
        emaPlay_set_led_array( LED_PATTERN_PAIR_ON, 
                               LED_PATTERN_PAIR_ON,
                               LED_PATTERN_PAIR_ON,
                               LED_PATTERN_PAIR_ON );
    } else if ( *sig == SIGNAL_GOOD ) {
        emaPlay_set_led_array( LED_PATTERN_PAIR_ON, 
                               LED_PATTERN_PAIR_ON,
                               LED_PATTERN_PAIR_ON,
                               LED_PATTERN_PAIR_OFF );
    } else if ( *sig == SIGNAL_FAIR ) {
        emaPlay_set_led_array( LED_PATTERN_PAIR_ON, 
                               LED_PATTERN_PAIR_ON,
                               LED_PATTERN_PAIR_OFF,
                               LED_PATTERN_PAIR_OFF );
    } else if ( *sig == SIGNAL_POOR ) {
        emaPlay_set_led_array( LED_PATTERN_PAIR_ON, 
                               LED_PATTERN_PAIR_OFF,
                               LED_PATTERN_PAIR_OFF,
                               LED_PATTERN_PAIR_OFF );
    }else{
        // signal none
        emaPlay_set_led_array( LED_PATTERN_PAIR_OFF, 
                               LED_PATTERN_PAIR_OFF,
                               LED_PATTERN_PAIR_OFF,
                               LED_PATTERN_PAIR_OFF );
    }
}

// indexs 0-3 = blu(d14-d17), 4-5 = red(d18-d19)
static uint16_t period_ticks[NUM_EMA_PLAY_LEDS];
static uint16_t duty_ticks[NUM_EMA_PLAY_LEDS];
static uint16_t period_cnt[NUM_EMA_PLAY_LEDS];
static uint16_t duty_cnt[NUM_EMA_PLAY_LEDS];

void set_led_blink_pattern(uint8_t idx, uint16_t duty_cycle, uint16_t period) 
{
    period_cnt[idx] = 0;
    duty_cnt[idx] = 0;
    duty_ticks[idx] = duty_cycle;
    period_ticks[idx] = period;
}

void update_led(uint8_t idx) 
{
    period_cnt[idx]++;
    duty_cnt[idx]++;
    uint8_t pin;

    // figure out gpio pin
    switch (idx) {
    case 0:
        pin = blu_led0;
        break;
    case 1:
        pin = blu_led1;
        break;
    case 2:
        pin = blu_led2;
        break;
    case 3:
        pin = blu_led3;
        break;
    case 4:
        pin = red_led0;
        break;
    case 5:
        pin = red_led1;
        break;
    default:
        break;
    }

    if (0 == duty_ticks[idx] && 0 == period_ticks[idx]) {
        // led off
        gpio_set_pin_level(pin, true);
    } else {

        // track the period
        if (period_cnt[idx] >= period_ticks[idx]) {
            // start over
            duty_cnt[idx] = 0;
            period_cnt[idx] = 0;
            gpio_set_pin_level(pin, false);
        }

        // track the duty cycle
        if (duty_cnt[idx] >= duty_ticks[idx]) {
            // turn off led
            gpio_set_pin_level(pin, true);
        }
    }
}

/*###########################################*/


