#include <atmel_start.h>
#include "emaPlay.h"
#include "oc_leaf.h"
#include <stdio.h>

int main(void)
{
    struct modem_at_context ema_modem_handle;
    struct ema_mgmt_context ema_mgmt_handle;
    static struct ema_status ema;
    static struct emaPlay_status emaPlay;

    // timers
    uint32_t system_task_exec_timer;
    uint32_t system_adc_scan_timer;
    uint32_t system_env_read_timer;

    // event status
    uint8_t ev_sts = 0;

    // Initializes MCU, drivers and middleware 
    atmel_start_init();
    // init the system 1ms tick time base
    emaPlay_init_sys1ms_tick( );

    // init q's for modem uart rx data
    emaPlay_q_init( &modem_uart_sync_rx_q, modem_uart_sync_rx_q_buff, sizeof(modem_uart_sync_rx_q_buff) );
    emaPlay_q_init( &modem_uart_async_rx_q, modem_uart_async_rx_q_buff, sizeof(modem_uart_async_rx_q_buff) );
    // init q's for mgmt uart rx data
    emaPlay_q_init( &mgmt_uart_sync_rx_q, mgmt_uart_sync_rx_q_buff, sizeof(mgmt_uart_sync_rx_q_buff) );
    emaPlay_q_init( &mgmt_uart_async_rx_q, mgmt_uart_async_rx_q_buff, sizeof(mgmt_uart_async_rx_q_buff) );

    // complete setup of the modem uart 
    emaPlay_uart_setup( &ema_modem_uart, &ema_modem_uart_io, (usart_cb_t)emaPlay_modem_tx_cb, (usart_cb_t)emaPlay_modem_rx_cb );
    // complete setup of the mgmt uart 
    emaPlay_uart_setup( &ema_mgmt_uart, &ema_mgmt_uart_io, (usart_cb_t)emaPlay_mgmt_tx_cb, (usart_cb_t)emaPlay_mgmt_rx_cb );

    // set the AT send/receive handler for the modem uart
    emaModemAT_Init( emaPlay_modem_at_query, emaPlay_modem_async_recv, &ema_modem_handle );
    // set the AT send/receive handler for the mgmt uart
    emaMgmtAT_Init( emaPlay_mgmt_at_query, &ema_mgmt_handle );

    // init the cell signal leds 
    emaPlay_set_led_array( LED_PATTERN_PAIR_SEARCHING, 
                           LED_PATTERN_PAIR_OFF,
                           LED_PATTERN_PAIR_OFF,
                           LED_PATTERN_PAIR_OFF );
    TIME_DELAY_ms(500);

    // init the debug input/output 
    #ifdef DEBUG
    uint32_t dbg_timer;
    emaPlay_q_init( &dbg_rx_q, dbg_q_rx_buff, sizeof(dbg_q_rx_buff) );
    emaPlay_q_init( &dbg_tx_q, dbg_q_tx_buff, sizeof(dbg_q_tx_buff) );
    emaPlay_uart_setup( &emaPlay_debug_uart, &emaPlay_dbg_uart_io, (usart_cb_t)emaPlay_dbg_tx_cb, (usart_cb_t)emaPlay_dbg_rx_cb );
    usart_async_enable( &emaPlay_debug_uart );
    emaPlay_dbg_uart_send( "\r\nema:Play \"Getting Started with ema\" FW Version: ", sizeof("\r\nema:Play \"Getting Started with ema\" FW Version: "), 20, 0 );
    emaPlay_dbg_uart_send( __DATE__, sizeof(__DATE__), 20, 0 );
    emaPlay_dbg_uart_send( "\r\n'?' for menu\r\n", sizeof("\r\n'?' for menu\r\n"), 20, 0 );
    dbg_timer = TIME_ELAPSED_ms(0);
    #endif

    // init timers
    system_task_exec_timer = TIME_ELAPSED_ms(0);
    system_adc_scan_timer = TIME_ELAPSED_ms(0);
    system_env_read_timer = TIME_ELAPSED_ms(0);

    //init adc
    adc_sync_enable_channel( &ADC_0, 0 );

    // init i2c for sht30 sensor
	i2c_m_sync_set_slaveaddr( &I2C, 0x0045, I2C_M_SEVEN );
	i2c_m_sync_enable( &I2C );

    // main application loop
    while (1) {

        // handle any asynch URCs, this must be called frequently and periodically
        emaPlay_mgmt_URC_handler( &ema_mgmt_handle );

        // execute the main system task every SYS_TASK_EXECUTION_INTERVAL
        if ( TIME_ELAPSED_ms(system_task_exec_timer) >= SYS_TASK_EXECUTION_INTERVAL || ev_sts ){
            system_task_exec_timer = TIME_ELAPSED_ms(0);
            emaPlay_system_task( &ema_mgmt_handle, &ema_modem_handle, &ema, &emaPlay, &ev_sts );
        }

        // scan the adc input every ADC_SCAN_INTERVAL_MS
        if ( TIME_ELAPSED_ms(system_adc_scan_timer) >= ADC_SCAN_INTERVAL_MS ){
            system_adc_scan_timer = TIME_ELAPSED_ms(0);
            emaPlay_scan_analog_inputs( &emaPlay );
        }

        // get the environment temp/hum every ENV_READ_INTERVAL
        if ( TIME_ELAPSED_ms(system_env_read_timer) >= ENV_READ_INTERVAL ){
            system_env_read_timer = TIME_ELAPSED_ms(0);
            emaPlay_read_temp_humidiity( &emaPlay );
        }

        // process any events that occured
        ev_sts = emaPlay_check_events( &emaPlay );

        // process debug input/ouput
        #ifdef DEBUG
        if ( TIME_ELAPSED_ms(dbg_timer) >= DBG_TASK_INTERVAL ){
            dbg_timer = TIME_ELAPSED_ms(0);
            dbg_task( &ema_mgmt_handle, &ema_modem_handle );
        }
        #endif
    }
}
